<?php
// admin/upload_image.php
// Robust TinyMCE image upload with clear JSON errors

if (session_status() === PHP_SESSION_NONE) { session_start(); }

// show errors as JSON so you can see the real cause
ini_set('display_errors', 0);
error_reporting(E_ALL);

// catch fatals too and return JSON
register_shutdown_function(function () {
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'PHP fatal', 'detail' => $e['message'], 'file' => $e['file'], 'line' => $e['line']]);
  }
});

function json_error($code, $msg) {
  http_response_code($code);
  header('Content-Type: application/json');
  echo json_encode(['error' => $msg]);
  exit;
}

// must be admin
if (empty($_SESSION['is_admin'])) {
  json_error(403, 'Forbidden, admin only');
}

// CSRF
if (empty($_POST['csrf_token']) || empty($_SESSION['csrf_token']) ||
    !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
  json_error(400, 'Invalid request, CSRF failed');
}

// file presence and error mapping
if (empty($_FILES['file'])) {
  json_error(400, 'No file part');
}
$err = $_FILES['file']['error'];
if ($err !== UPLOAD_ERR_OK) {
  $map = [
    UPLOAD_ERR_INI_SIZE   => 'File exceeds upload_max_filesize',
    UPLOAD_ERR_FORM_SIZE  => 'File exceeds form MAX_FILE_SIZE',
    UPLOAD_ERR_PARTIAL    => 'Partial upload',
    UPLOAD_ERR_NO_FILE    => 'No file uploaded',
    UPLOAD_ERR_NO_TMP_DIR => 'Missing temp directory',
    UPLOAD_ERR_CANT_WRITE => 'Cannot write to disk',
    UPLOAD_ERR_EXTENSION  => 'Blocked by PHP extension'
  ];
  json_error(400, $map[$err] ?? ('Upload error code ' . $err));
}

$uploadDir = dirname(__DIR__) . '/assets/uploads/';
if (!is_dir($uploadDir) && !@mkdir($uploadDir, 0755, true)) {
  json_error(500, 'Could not create uploads directory');
}

// size limit 5 MB
$size = (int)$_FILES['file']['size'];
if ($size <= 0)                  json_error(400, 'Empty file');
if ($size > 5 * 1024 * 1024)     json_error(400, 'File too large, max 5 MB');

$tmp  = $_FILES['file']['tmp_name'];
$name = $_FILES['file']['name'];

// mime check with fallbacks
$allowedExt  = ['jpg','jpeg','png','gif','webp'];
$allowedMime = ['image/jpeg','image/png','image/gif','image/webp'];

$mime = null;
if (class_exists('finfo')) {
  $finfo = new finfo(FILEINFO_MIME_TYPE);
  $mime  = $finfo->file($tmp);
} elseif (function_exists('mime_content_type')) {
  $mime = mime_content_type($tmp);
} else {
  $imgInfo = @getimagesize($tmp);
  $mime = $imgInfo['mime'] ?? null;
}
if (!$mime || !in_array($mime, $allowedMime, true)) {
  json_error(400, 'Unsupported file type, got ' . ($mime ?: 'unknown'));
}

// safe filename
$ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
if (!in_array($ext, $allowedExt, true)) {
  $map = ['image/jpeg'=>'jpg','image/png'=>'png','image/gif'=>'gif','image/webp'=>'webp'];
  $ext = $map[$mime] ?? 'jpg';
}
$fname = date('Ymd_His') . '_' . bin2hex(random_bytes(8)) . '.' . $ext;
$dest  = $uploadDir . $fname;

if (!is_uploaded_file($tmp)) {
  json_error(400, 'Upload not valid');
}
if (!@move_uploaded_file($tmp, $dest)) {
  json_error(500, 'Could not save file, check folder permissions on assets/uploads');
}

// build public URL from current script path
// build absolute public URL like http://localhost/MaColX_Starter/assets/uploads/filename.jpg
$scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');   // /MaColX_Starter/admin
$root      = rtrim(dirname($scriptDir), '/\\');                // /MaColX_Starter
if ($root === '/' || $root === '\\') { $root = ''; }

$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$host   = $_SERVER['HTTP_HOST'];

$publicUrl = $scheme . '://' . $host . $root . '/assets/uploads/' . rawurlencode($fname);



// success
header('Content-Type: application/json');
echo json_encode(['location' => $publicUrl]);
