<?php
// Start session early for CSRF token
if (session_status() === PHP_SESSION_NONE) {
  session_start();
}

$page_title = "Contact — MaColX";

// Create CSRF token if missing
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

include __DIR__ . "/includes/header.php";
$pdo = db();

$msg = "";
$msg_type = "info";

// Sticky values
$name = $email = $subject = $message = "";

// Handle submit
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $pdo) {
  // CSRF check (compare POST token to session token)
  $token_ok = isset($_POST['csrf_token'], $_SESSION['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token']);

  // Honeypot, must stay empty
  $honeypot = trim($_POST['website'] ?? '');

  $name    = trim($_POST['name'] ?? '');
  $email   = trim($_POST['email'] ?? '');
  $subject = trim($_POST['subject'] ?? '');
  $message = trim($_POST['message'] ?? '');

  if (!$token_ok) {
    $msg = "Security check failed. Please reload the page and try again.";
    $msg_type = "danger";
  } elseif ($honeypot !== "") {
    // Silently treat as success to bots, but do nothing
    $msg = "Thank you. Your message has been received.";
    $msg_type = "info";
  } elseif ($name && filter_var($email, FILTER_VALIDATE_EMAIL) && $subject && $message) {
    try {
      // 1) Save to DB
      $stmt = $pdo->prepare("INSERT INTO messages(name,email,subject,message) VALUES(?,?,?,?)");
      $stmt->execute([$name, $email, $subject, $message]);

      // 2) Send email notification
      // -----------------------------------
      // Basic hardening for header values
      $safe_name    = str_replace(["\r", "\n"], ' ', $name);
      $safe_subject = str_replace(["\r", "\n"], ' ', $subject);

      // Recipient (your email)
      $recipient = "acomaruiso1@gmail.com";

      // Subject line for you
      $email_subject = "[MaColX Website] " . mb_substr($safe_subject, 0, 120);

      // Build message body
      $email_body = "You have received a new contact message from the MaColX website.\n\n"
        . "Name: " . $safe_name . "\n"
        . "Email: " . $email . "\n"
        . "Subject: " . $safe_subject . "\n\n"
        . "Message:\n"
        . $message . "\n\n"
        . "---- Meta ----\n"
        . "IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown') . "\n"
        . "Date: " . date('Y-m-d H:i:s') . "\n";

      // From address based on server host
      $host = $_SERVER['SERVER_NAME'] ?? 'example.com';
      $host = preg_replace('/^www\./i', '', $host);
      $from_address = "no-reply@" . $host;

      $headers  = "From: MaColX Website <" . $from_address . ">\r\n";
      $headers .= "Reply-To: " . $email . "\r\n";
      $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";

      // Attempt to send (errors here won't affect user success message)
      @mail($recipient, $email_subject, $email_body, $headers);
      // -----------------------------------

      // 3) User-facing success
      $msg = "Thank you. Your message has been received.";
      $msg_type = "success";

      // Clear sticky values on success
      $name = $email = $subject = $message = "";
      // Rotate CSRF token after success
      $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    } catch (Exception $e) {
      $msg = "Could not save your message. Please try again.";
      $msg_type = "danger";
    }
  } else {
    $msg = "Please fill all fields with a valid email.";
    $msg_type = "warning";
  }
}
?>

<style>
/* ===== Contact page specific styling (kept small and scoped) ===== */

.section-contact-hero{
  position: relative;
  padding-top: 3.0rem;
  padding-bottom: 3.2rem;
  overflow: hidden;
  background:
    radial-gradient(900px 520px at 0% -20%, rgba(37,99,235,0.18), transparent 60%),
    radial-gradient(900px 520px at 100% -10%, rgba(16,185,129,0.14), transparent 60%),
    linear-gradient(to bottom, #020617, #020617 40%, #030712 100%);
  color: #e5e7eb;
}

.section-contact-hero > .container{
  position: relative;
  z-index: 2;
}

.contact-particles-canvas{
  position: absolute;
  inset: 0;
  width: 100%;
  height: 100%;
  z-index: 1;
  pointer-events: none;
  opacity: 0.55;
}

.section-contact-hero .lead{
  color: rgba(226,232,240,0.9);
}

.section-contact-main{
  padding-top: 2.2rem;
  padding-bottom: 3.0rem;
}

/* Contact form card animation */
.mx-contact-form-card{
  position: relative;
  border-radius: 22px;
  border: 1px solid rgba(148,163,184,0.55);
  box-shadow:
    0 16px 40px rgba(15,23,42,0.10),
    0 0 0 1px rgba(255,255,255,0.7) inset;
  transition:
    transform .22s ease,
    box-shadow .22s ease,
    border-color .22s ease;
}

.mx-contact-form-card:hover{
  transform: translateY(-3px);
  box-shadow:
    0 20px 55px rgba(15,23,42,0.16),
    0 0 0 1px rgba(59,130,246,0.18);
}

/* Success pulse */
@keyframes contactCardGlow{
  0%{
    box-shadow:
      0 0 0 rgba(59,130,246,0),
      0 16px 40px rgba(15,23,42,0.14);
  }
  50%{
    box-shadow:
      0 0 0 14px rgba(59,130,246,0.18),
      0 22px 55px rgba(15,23,42,0.22);
  }
  100%{
    box-shadow:
      0 0 0 rgba(59,130,246,0),
      0 16px 40px rgba(15,23,42,0.14);
  }
}
.mx-contact-form-card.form-success-pulse{
  animation: contactCardGlow 1.6s ease-out 1;
}

/* Button states */
#contactSubmitBtn.is-sending{
  opacity: .85;
  transform: translateY(1px);
}
#contactSubmitBtn.is-success{
  background:
    radial-gradient(circle at 20% 0%, #34d399 0, #16a34a 40%, #15803d 100%);
  border-color: rgba(22,163,74,0.75);
}

/* Spinner visibility (Bootstrap's .d-none used) */
#contactSpinner{
  transition: opacity .15s ease;
}

/* Contact meta info */
.contact-meta-card{
  border-radius: 20px;
}

/* Dark mode tweaks */
@media (prefers-color-scheme: dark){
  .section-contact-hero{
    background:
      radial-gradient(900px 520px at 0% -20%, rgba(37,99,235,0.30), transparent 60%),
      radial-gradient(900px 520px at 100% -10%, rgba(16,185,129,0.26), transparent 60%),
      linear-gradient(to bottom, #020617, #020617 40%, #000 100%);
  }
}

@media (max-width: 575.98px){
  .section-contact-hero{
    padding-top: 2.6rem;
    padding-bottom: 2.6rem;
  }
}

/* =========================
   FIX: Contact hero response box text contrast
   ========================= */
.section-contact-hero .mx-glass {
  background: radial-gradient(circle at 0 0, rgba(59,130,246,0.18), rgba(15,23,42,0.85));
  border: 1px solid rgba(148,163,184,0.65);
  box-shadow:
    0 10px 25px rgba(0,0,0,0.6),
    0 0 0 1px rgba(59,130,246,0.25);
}

.section-contact-hero .mx-glass .text-muted {
  color: rgba(229,231,235,0.88) !important;  /* brighter gray */
}

.section-contact-hero .mx-glass .fw-semibold {
  color: #ffffff;  /* full white for title text */
}

.section-contact-hero .mx-glass .text-uppercase {
  color: #93c5fd !important; /* subtle blue highlight for header */
  letter-spacing: .12em;
}

/* =========================
   ANIMATED GLOW FOR "Typical Response" BOX
   ========================= */
.section-contact-hero .mx-glass {
  position: relative;
  overflow: hidden;
  animation: contactBoxPulse 6s ease-in-out infinite;
}

@keyframes contactBoxPulse {
  0% {
    box-shadow:
      0 10px 25px rgba(59,130,246,0.25),
      0 0 0 0 rgba(59,130,246,0.0);
    transform: scale(1);
  }
  50% {
    box-shadow:
      0 12px 35px rgba(37,99,235,0.45),
      0 0 30px rgba(16,185,129,0.25);
    transform: scale(1.01);
  }
  100% {
    box-shadow:
      0 10px 25px rgba(59,130,246,0.25),
      0 0 0 0 rgba(59,130,246,0.0);
    transform: scale(1);
  }
}

/* Optional soft shimmer line across top of the box */
.section-contact-hero .mx-glass::before {
  content: "";
  position: absolute;
  top: 0;
  left: -50%;
  width: 200%;
  height: 1px;
  background: linear-gradient(90deg,
    transparent,
    rgba(59,130,246,0.35),
    rgba(16,185,129,0.25),
    transparent);
  animation: shimmerSweep 8s linear infinite;
}

@keyframes shimmerSweep {
  0% { transform: translateX(-10%); opacity: .3; }
  50% { transform: translateX(10%); opacity: .7; }
  100% { transform: translateX(-10%); opacity: .3; }
}
</style>

<!-- HERO + PARTICLE BACKGROUND -->
<section class="section-contact-hero">
  <canvas id="contactParticles" class="contact-particles-canvas"></canvas>

  <div class="container">
    <div class="row align-items-center g-4">
      <div class="col-lg-6">
        <span class="mx-role-chip mb-2">Contact MaColX</span>
        <h1 class="h1 mb-2 mt-2">Let’s build better digital health systems.</h1>
        <p class="lead mb-3">
          Reach out to discuss digital health architectures, data science collaborations,
          and implementation support for Ghana and beyond.
        </p>
        <ul class="list-unstyled small mb-0">
          <li class="mb-1"><i class="bi bi-geo-alt me-1"></i> Ghana Health Service HQ, Accra</li>
          <li class="mb-1"><i class="bi bi-envelope-open me-1"></i> marius.acolatse@ghs.gov.gh</li>
        </ul>
      </div>
      <div class="col-lg-6 text-center text-lg-end">
        <div class="mx-glass p-3 p-md-4 d-inline-block">
          <div class="d-flex align-items-center gap-3">
            <div>
              <div class="metrics-dot-live"></div>
            </div>
            <div class="text-start small">
              <div class="text-uppercase ls-wide mb-1" style="font-size:.7rem;">Typical response</div>
              <div class="fw-semibold">Within 1–2 business days</div>
              <div class="text-muted">We prioritise projects aligned with public health impact.</div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>
</section>

<!-- MAIN CONTACT SECTION -->
<section class="section section-contact-main">
  <div class="container">
    <div class="row g-4 align-items-start">
      
      <!-- Left: Contact info / reassurance -->
      <div class="col-lg-4">
        <div class="contact-meta-card mx-glass p-3 p-md-4 h-100">
          <h2 class="h5 mb-2">Work with MaColX</h2>
          <p class="small text-muted mb-3">
            Share a brief overview of your programme, timelines, and priority use-cases.
            We’ll follow up with clarifying questions or a short discovery call.
          </p>
          <ul class="list-unstyled small mb-3">
            <li class="mb-2">
              <i class="bi bi-shield-lock me-1"></i>
              We treat submissions in line with data privacy best practices.
            </li>
            <li class="mb-2">
              <i class="bi bi-diagram-3 me-1"></i>
              Ideal for ministries, NGOs, multilaterals, and research partners.
            </li>
            <li class="mb-0">
              <i class="bi bi-file-earmark-text me-1"></i>
              You’ll receive a structured response, not generic sales copy.
            </li>
          </ul>
        </div>
      </div>

      <!-- Right: Form -->
      <div class="col-lg-8">
        <div class="mx-contact-form-card mx-glass p-3 p-md-4">

          <?php if($msg): ?>
            <div class="alert alert-<?php echo htmlspecialchars($msg_type); ?> shadow-sm mb-3"
                 <?php if($msg_type === 'success') echo 'data-form-status="success"'; ?>>
              <?php echo htmlspecialchars($msg); ?>
            </div>
          <?php endif; ?>

          <form id="contactForm" method="post" class="row g-3" novalidate>
            <!-- CSRF -->
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
            <!-- Honeypot, keep hidden -->
            <div style="display:none;">
              <label>Website</label>
              <input type="text" name="website" autocomplete="off">
            </div>

            <div class="col-md-6">
              <label class="form-label">Name</label>
              <input name="name"
                     class="form-control"
                     required
                     value="<?php echo htmlspecialchars($name); ?>">
            </div>

            <div class="col-md-6">
              <label class="form-label">Email</label>
              <input name="email"
                     type="email"
                     class="form-control"
                     required
                     value="<?php echo htmlspecialchars($email); ?>">
            </div>

            <div class="col-12">
              <label class="form-label">Subject</label>
              <input name="subject"
                     class="form-control"
                     required
                     placeholder="e.g. Digital health architecture support for CHPS"
                     value="<?php echo htmlspecialchars($subject); ?>">
            </div>

            <div class="col-12">
              <label class="form-label">Message</label>
              <textarea name="message"
                        rows="5"
                        class="form-control"
                        required
                        placeholder="Tell us about your programme, timelines, and key challenges."><?php echo htmlspecialchars($message); ?></textarea>
            </div>

            <div class="col-12 d-flex align-items-center gap-2">
              <button id="contactSubmitBtn" class="btn btn-mx" type="submit">
                <!-- Spinner (hidden by default, shown while sending) -->
                <span id="contactSpinner"
                      class="spinner-border spinner-border-sm me-1 d-none"
                      role="status"
                      aria-hidden="true"></span>
                <span class="btn-label-text">Send message</span>
              </button>
              <a class="btn btn-ghost" href="<?php echo $base; ?>/index.php">Back to home</a>
            </div>
          </form>

        </div>
      </div>

    </div>
  </div>
</section>

<!-- Small standards reassurance strip -->
<section class="section-tight text-center">
  <div class="container">
    <p class="small text-muted mb-0">
      MaColX work is aligned with <strong>ICD-10/ICD-11</strong>, <strong>HL7 FHIR</strong>,
      and <strong>WHO digital health guidelines</strong>, with strict data governance.
    </p>
  </div>
</section>

<script>
// ===============================
// A) FORM MOTION FEEDBACK + SPINNER
// ===============================
document.addEventListener('DOMContentLoaded', function () {
  const form = document.getElementById('contactForm');
  const submitBtn = document.getElementById('contactSubmitBtn');
  const btnLabel = submitBtn ? submitBtn.querySelector('.btn-label-text') : null;
  const spinner = document.getElementById('contactSpinner');
  const alertSuccess = document.querySelector('[data-form-status="success"]');
  const formCard = document.querySelector('.mx-contact-form-card');

  // On submit: show "sending" state + spinner
  if (form && submitBtn && btnLabel) {
    form.addEventListener('submit', function () {
      if (!submitBtn.classList.contains('is-sending')) {
        submitBtn.classList.add('is-sending');
        submitBtn.setAttribute('disabled', 'disabled');
        if (spinner) spinner.classList.remove('d-none');
        btnLabel.textContent = 'Sending…';
      }
    });
  }

  // On successful POST: pulse + success state
  if (alertSuccess && formCard && submitBtn && btnLabel) {
    // Card glow pulse
    formCard.classList.add('form-success-pulse');

    // Button success look
    submitBtn.classList.remove('is-sending');
    submitBtn.classList.add('is-success');
    submitBtn.removeAttribute('disabled');
    if (spinner) spinner.classList.add('d-none');
    btnLabel.textContent = 'Message sent ✔';
  }
});

// ===============================
// C) PARTICLE BACKGROUND (CANVAS)
// ===============================
(function () {
  const canvas = document.getElementById('contactParticles');
  if (!canvas) return;

  const ctx = canvas.getContext('2d');
  const heroSection = document.querySelector('.section-contact-hero');
  let particles = [];
  const PARTICLE_COUNT = 45;

  function resizeCanvas() {
    if (!heroSection) return;
    const rect = heroSection.getBoundingClientRect();
    const ratio = window.devicePixelRatio || 1;
    canvas.width = rect.width * ratio;
    canvas.height = rect.height * ratio;
    ctx.setTransform(ratio, 0, 0, ratio, 0, 0);
  }

  function createParticles() {
    particles = [];
    const rect = heroSection.getBoundingClientRect();
    for (let i = 0; i < PARTICLE_COUNT; i++) {
      particles.push({
        x: Math.random() * rect.width,
        y: Math.random() * rect.height,
        vx: (Math.random() - 0.5) * 0.35,
        vy: (Math.random() - 0.5) * 0.35,
        r: 1.5 + Math.random() * 1.4
      });
    }
  }

  function draw() {
    if (!heroSection) return;
    const rect = heroSection.getBoundingClientRect();
    ctx.clearRect(0, 0, rect.width, rect.height);

    // Draw connections
    for (let i = 0; i < particles.length; i++) {
      for (let j = i + 1; j < particles.length; j++) {
        const dx = particles[i].x - particles[j].x;
        const dy = particles[i].y - particles[j].y;
        const dist = Math.sqrt(dx * dx + dy * dy);
        if (dist < 120) {
          const alpha = 1 - dist / 120;
          ctx.beginPath();
          ctx.moveTo(particles[i].x, particles[i].y);
          ctx.lineTo(particles[j].x, particles[j].y);
          ctx.strokeStyle = 'rgba(148, 163, 184,' + (0.18 * alpha) + ')';
          ctx.lineWidth = 0.8;
          ctx.stroke();
        }
      }
    }

    // Draw particles
    for (const p of particles) {
      ctx.beginPath();
      ctx.arc(p.x, p.y, p.r, 0, Math.PI * 2);
      ctx.fillStyle = 'rgba(96, 165, 250, 0.8)';
      ctx.fill();

      // Slight glow halo
      ctx.beginPath();
      ctx.arc(p.x, p.y, p.r + 2, 0, Math.PI * 2);
      ctx.strokeStyle = 'rgba(37, 99, 235, 0.20)';
      ctx.lineWidth = 0.4;
      ctx.stroke();

      // Move
      p.x += p.vx;
      p.y += p.vy;

      // Wrap around edges
      if (p.x < -10) p.x = rect.width + 10;
      if (p.x > rect.width + 10) p.x = -10;
      if (p.y < -10) p.y = rect.height + 10;
      if (p.y > rect.height + 10) p.y = -10;
    }

    requestAnimationFrame(draw);
  }

  function init() {
    resizeCanvas();
    createParticles();
    requestAnimationFrame(draw);
  }

  window.addEventListener('resize', function () {
    resizeCanvas();
    createParticles();
  });

  // Start when layout is ready
  window.setTimeout(init, 150);
})();
</script>

<?php include __DIR__ . "/includes/footer.php"; ?>
